<?php

/**
 * Sentinel Event Definitions
 *
 * Centralized event registry for all default system events.
 * This file is the single source of truth for event definitions.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get all default system event definitions
 *
 * @return array Array of event definitions with keys as event names
 */
function sentinel_get_default_event_definitions()
{
    return array(
        // Authentication Events
        'user_login' => array(
            'category' => 'authentication',
            'priority' => 'low',
            'description' => 'User successfully logged in'
        ),
        'user_logout' => array(
            'category' => 'authentication',
            'priority' => 'low',
            'description' => 'User logged out'
        ),
        'user_registered' => array(
            'category' => 'authentication',
            'priority' => 'low',
            'description' => 'New user registered'
        ),
        'login_failed' => array(
            'category' => 'authentication',
            'priority' => 'medium',
            'description' => 'Failed login attempt'
        ),
        'auth_cookie_bad' => array(
            'category' => 'authentication',
            'priority' => 'medium',
            'description' => 'Invalid authentication cookie'
        ),

        // Premium: Authentication attempt tracking
        'auth_success' => array(
            'category' => 'authentication',
            'priority' => 'low',
            'description' => 'Successful authentication attempt'
        ),
        'auth_fail' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Failed authentication attempt'
        ),

        // Premium: Security incident events
        'security_incident_opened' => array(
            'category' => 'security',
            'priority' => 'high',
            'description' => 'Security incident detected and opened'
        ),
        'security_incident_updated' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Security incident updated or escalated'
        ),
        'security_incident_resolved' => array(
            'category' => 'security',
            'priority' => 'low',
            'description' => 'Security incident automatically resolved'
        ),

        // Content Events
        'profile_updated' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'User profile updated'
        ),
        'post_published' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'Post published'
        ),
        'post_updated' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'Post updated'
        ),
        'post_deleted' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'Post deleted'
        ),
        'comment_posted' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'Comment posted'
        ),
        'comment_approved' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'Comment approved'
        ),
        'comment_deleted' => array(
            'category' => 'content',
            'priority' => 'low',
            'description' => 'Comment deleted'
        ),

        // System Events
        'plugin_activated' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Plugin activated'
        ),
        'plugin_deactivated' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Plugin deactivated'
        ),
        'plugin_updated' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Plugin updated to new version'
        ),
        'plugin_events_cleaned' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Plugin events cleaned up'
        ),
        'theme_switched' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Site theme was changed'
        ),
        'theme_updated' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Theme updated to new version'
        ),
        'core_upgraded' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'WordPress core was upgraded'
        ),
        'maintenance_mode_enabled' => array(
            'category' => 'system',
            'priority' => 'medium',
            'description' => 'Site entered maintenance mode'
        ),
        'maintenance_mode_disabled' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Site exited maintenance mode'
        ),
        'api_key_generated' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'API key generated or regenerated'
        ),
        'option_updated' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'WordPress option or setting was changed'
        ),
        'logs_cleaned' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'System cleaned up old logs'
        ),
        'tables_optimized' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Database tables optimized'
        ),
        'admin_test' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Admin test event'
        ),

        // Rate limiting events
        'rate_limit_warning' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Rate limit warning threshold reached'
        ),
        'rate_limit_exceeded' => array(
            'category' => 'security',
            'priority' => 'high',
            'description' => 'Rate limit exceeded'
        ),
        'security_action_triggered' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Security action was triggered'
        ),
        'security_throttle_applied' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Security throttle was applied'
        ),
        'security_ip_blocked' => array(
            'category' => 'security',
            'priority' => 'high',
            'description' => 'IP address was temporarily blocked'
        ),
        'security_blocked_attempt' => array(
            'category' => 'security',
            'priority' => 'high',
            'description' => 'Blocked attempt from restricted IP'
        ),

        // File monitoring events
        'file_modified' => array(
            'category' => 'security',
            'priority' => 'high',
            'description' => 'File was modified or changed'
        ),

        // Error Events
        'php_fatal_error' => array(
            'category' => 'errors',
            'priority' => 'critical',
            'description' => 'PHP fatal error occurred'
        ),
        'php_warning' => array(
            'category' => 'errors',
            'priority' => 'medium',
            'description' => 'PHP warning occurred'
        ),
        'php_notice' => array(
            'category' => 'errors',
            'priority' => 'low',
            'description' => 'PHP notice occurred'
        ),
        'php_deprecated' => array(
            'category' => 'errors',
            'priority' => 'medium',
            'description' => 'PHP deprecated function used'
        ),
        'wp_database_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'WordPress database error'
        ),
        'wp_memory_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'WordPress memory limit exceeded'
        ),
        'wp_permission_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'WordPress permission error'
        ),
        'wp_config_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'WordPress configuration error'
        ),
        'nonce_verification_failed' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Nonce verification failed'
        ),
        'permission_denied' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Permission denied'
        ),
        'http_404_error' => array(
            'category' => 'errors',
            'priority' => 'low',
            'description' => 'HTTP 404 error (page not found)'
        ),
        'http_500_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'HTTP 500 error (server error)'
        ),
        'wp_remote_error' => array(
            'category' => 'errors',
            'priority' => 'medium',
            'description' => 'WordPress remote request error'
        ),
        'wp_cron_error' => array(
            'category' => 'errors',
            'priority' => 'medium',
            'description' => 'WordPress cron error'
        ),
        'plugin_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'Plugin error occurred'
        ),
        'theme_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'Theme error occurred'
        ),
        'plugin_activation_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'Plugin activation error'
        ),
        'file_upload_error' => array(
            'category' => 'errors',
            'priority' => 'medium',
            'description' => 'File upload error'
        ),
        'file_permission_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'File permission error'
        ),

        // Additional WordPress errors
        'wp_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'Generic WordPress error'
        ),

        // Audit/performance events
        'log_accessed' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Log file accessed'
        ),
        'memory_peak_usage' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Memory peak usage recorded'
        ),
        'db_query_count' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Database query count recorded'
        ),
        'response_time_ms' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Response time recorded'
        ),
        'memory_early_warning' => array(
            'category' => 'system',
            'priority' => 'medium',
            'description' => 'Memory usage early warning'
        ),
        'memory_warning' => array(
            'category' => 'system',
            'priority' => 'medium',
            'description' => 'Memory usage warning'
        ),
        'memory_critical' => array(
            'category' => 'system',
            'priority' => 'critical',
            'description' => 'Critical memory usage'
        ),
        'memory_auto_optimization' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Automatic memory optimization performed'
        ),
        'memory_leak_warning' => array(
            'category' => 'system',
            'priority' => 'medium',
            'description' => 'Potential memory leak detected'
        ),

        // Privacy & GDPR events
        'user_deletion_request' => array(
            'category' => 'privacy',
            'priority' => 'low',
            'description' => 'User deletion request submitted'
        ),
        'user_data_anonymized' => array(
            'category' => 'privacy',
            'priority' => 'low',
            'description' => 'User data anonymized'
        ),
        'admin_deletion_request_processed' => array(
            'category' => 'privacy',
            'priority' => 'low',
            'description' => 'Admin processed user deletion request'
        ),

        // Email events
        'email_sent' => array(
            'category' => 'system',
            'priority' => 'low',
            'description' => 'Email sent successfully'
        ),
        'email_failed' => array(
            'category' => 'errors',
            'priority' => 'medium',
            'description' => 'Email failed to send'
        ),
        'email_delivery_error' => array(
            'category' => 'errors',
            'priority' => 'high',
            'description' => 'Email delivery error'
        ),
        'email_spam_detected' => array(
            'category' => 'security',
            'priority' => 'medium',
            'description' => 'Spam email detected'
        )
    );
}

/**
 * Get list of default event keys (for backward compatibility)
 *
 * @return array Array of event keys
 */
function sentinel_get_default_event_keys()
{
    return array_keys(sentinel_get_default_event_definitions());
}
