<?php
/**
 * Sentinel Events Class
 * 
 * Handles event registration and management.
 * This class acts as a registry for all events that can be logged.
 * Think of it as a catalog of everything we can track.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Sentinel_Events {
    
    /**
     * Array to store all registered events
     * This is where we keep track of what events exist and their configuration
     */
    private static $events = array();
    
    /**
     * Constructor - not much needed here since we use static methods
     */
    public function __construct() {
        // Constructor is mostly empty since we use static methods
    }
    
    /**
     * Register a new event that can be logged
     * This is how we tell the system about a new type of event to track
     * 
     * @param string $event_key Unique identifier for the event (e.g., 'user_login')
     * @param array $config Configuration array for the event
     * @return bool True if event was registered successfully
     */
    public static function register_event($event_key, $config) {
        // Validate the event key - must be a string and not empty
        if (empty($event_key) || !is_string($event_key)) {
            return false;
        }
        
        // Set default configuration values
        $default_config = array(
            'category' => 'general',        // What type of event (auth, content, system, etc.)
            'priority' => 'medium',         // How important (low, medium, high, critical)
            'auto_capture' => false,        // Should this be captured automatically via hooks
            'hook' => null,                 // WordPress hook to listen for (if auto_capture)
            'description' => '',            // Human-readable description
            'data_fields' => array(),       // What data fields this event can have
            'enabled' => true               // Whether this event is currently enabled
        );
        
        // Merge provided config with defaults
        $final_config = array_merge($default_config, $config);
        
        // Validate priority
        $valid_priorities = array('low', 'medium', 'high', 'critical');
        if (!in_array($final_config['priority'], $valid_priorities)) {
            $final_config['priority'] = 'medium';
        }
        
        // Validate category
        $valid_categories = array('authentication', 'content', 'system', 'security', 'user', 'admin', 'general', 'audit');
        if (!in_array($final_config['category'], $valid_categories)) {
            $final_config['category'] = 'general';
        }
        
        // Store the event configuration
        self::$events[$event_key] = $final_config;
        
        // Allow other code to hook into event registration
        do_action('sentinel_event_registered', $event_key, $final_config);
        
        return true;
    }
    
    /**
     * Get configuration for a specific event
     * 
     * @param string $event_key The event to get configuration for
     * @return array|null Event configuration or null if not found
     */
    public static function get_event($event_key) {
        return isset(self::$events[$event_key]) ? self::$events[$event_key] : null;
    }
    
    /**
     * Get all events in a specific category
     * 
     * @param string $category The category to filter by
     * @return array Array of events in that category
     */
    public static function get_events_by_category($category) {
        $filtered_events = array();
        
        foreach (self::$events as $event_key => $config) {
            if ($config['category'] === $category) {
                $filtered_events[$event_key] = $config;
            }
        }
        
        return $filtered_events;
    }
    
    /**
     * Get all registered events
     * 
     * @return array All registered events
     */
    public static function get_all_events() {
        return self::$events;
    }
    
    /**
     * Check if an event is registered
     * 
     * @param string $event_key The event to check
     * @return bool True if event exists and is enabled
     */
    public static function is_event_registered($event_key) {
        return isset(self::$events[$event_key]) && self::$events[$event_key]['enabled'];
    }
    
    /**
     * Enable or disable an event
     * 
     * @param string $event_key The event to modify
     * @param bool $enabled Whether to enable or disable the event
     * @return bool True if event was found and modified
     */
    public static function set_event_status($event_key, $enabled) {
        if (isset(self::$events[$event_key])) {
            self::$events[$event_key]['enabled'] = (bool) $enabled;
            return true;
        }
        
        return false;
    }
    
    /**
     * Get events that should be captured automatically
     * These are events that have auto_capture set to true
     * 
     * @return array Array of auto-capture events
     */
    public static function get_auto_capture_events() {
        $auto_events = array();
        
        foreach (self::$events as $event_key => $config) {
            if ($config['auto_capture'] && $config['enabled']) {
                $auto_events[$event_key] = $config;
            }
        }
        
        return $auto_events;
    }
    
    /**
     * Get events by priority level
     * 
     * @param string $priority The priority level to filter by
     * @return array Array of events with that priority
     */
    public static function get_events_by_priority($priority) {
        $filtered_events = array();
        
        foreach (self::$events as $event_key => $config) {
            if ($config['priority'] === $priority) {
                $filtered_events[$event_key] = $config;
            }
        }
        
        return $filtered_events;
    }
    
    /**
     * Get event statistics
     * Returns counts of events by category and priority
     * 
     * @return array Statistics about registered events
     */
    public static function get_event_statistics() {
        $stats = array(
            'total_events' => count(self::$events),
            'by_category' => array(),
            'by_priority' => array(),
            'auto_capture_count' => 0,
            'enabled_count' => 0
        );
        
        foreach (self::$events as $event_key => $config) {
            // Count by category
            $category = $config['category'];
            if (!isset($stats['by_category'][$category])) {
                $stats['by_category'][$category] = 0;
            }
            $stats['by_category'][$category]++;
            
            // Count by priority
            $priority = $config['priority'];
            if (!isset($stats['by_priority'][$priority])) {
                $stats['by_priority'][$priority] = 0;
            }
            $stats['by_priority'][$priority]++;
            
            // Count auto-capture events
            if ($config['auto_capture']) {
                $stats['auto_capture_count']++;
            }
            
            // Count enabled events
            if ($config['enabled']) {
                $stats['enabled_count']++;
            }
        }
        
        return $stats;
    }
    
    /**
     * Clear all registered events
     * Useful for testing or resetting the registry
     */
    public static function clear_events() {
        self::$events = array();
    }
} 