<?php
/**
 * Sentinel Email Templates Class
 * 
 * Handles email template generation for digest reports and notifications.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Sentinel_Email_Templates {
    
    /**
     * Generate daily summary digest email
     * 
     * @param array $stats Daily statistics data
     * @return array Email data (subject, html_body, plain_body)
     */
    public static function generate_daily_summary_email($stats, $section_toggles = array()) {
        $date = date('F j, Y', strtotime($stats['date_range']['start']));
        $subject = "Sentinel Daily Summary - $date";
        
        // Plain text version
        $plain_body = "Sentinel Daily Summary Report\n";
        $plain_body .= "Date: $date\n";
        $plain_body .= "Generated: " . current_time('F j, Y g:i A') . "\n\n";
        
        // Always show overview
        $plain_body .= "📊 OVERVIEW\n";
        $plain_body .= "Total Events: " . $stats['total_events'] . "\n";
        $plain_body .= "Active Users: " . $stats['active_users'] . "\n\n";
        
        // Only show if enabled
        if (empty($section_toggles) || !empty($section_toggles['daily_summary'])) {
            if (!empty($stats['events_by_category'])) {
                $plain_body .= "📈 EVENTS BY CATEGORY\n";
                foreach ($stats['events_by_category'] as $category) {
                    $plain_body .= "- " . ucfirst($category->category) . ": " . $category->count . "\n";
                }
                $plain_body .= "\n";
            }
            if (!empty($stats['events_by_priority'])) {
                $plain_body .= "⚠️ EVENTS BY PRIORITY\n";
                foreach ($stats['events_by_priority'] as $priority) {
                    $plain_body .= "- " . ucfirst($priority->priority) . ": " . $priority->count . "\n";
                }
                $plain_body .= "\n";
            }
            if (!empty($stats['top_events'])) {
                $plain_body .= "🔥 TOP EVENTS\n";
                foreach (array_slice($stats['top_events'], 0, 5) as $event) {
                    $plain_body .= "- " . $event->event_key . ": " . $event->count . " times\n";
                }
                $plain_body .= "\n";
            }
        }
        
        $plain_body .= "View full report: " . admin_url('admin.php?page=sentinel-dashboard') . "\n";
        $plain_body .= "Generated by Sentinel - WordPress Activity Logger";
        
        // HTML version
        $html_body = self::get_email_header($subject);
        $html_body .= '<div class="content">';
        
        // Always show overview
        $html_body .= '<div class="overview-section">';
        $html_body .= '<h2>📊 Daily Overview</h2>';
        $html_body .= '<div class="stats-grid">';
        $html_body .= '<div class="stat-card">';
        $html_body .= '<div class="stat-number">' . $stats['total_events'] . '</div>';
        $html_body .= '<div class="stat-label">Total Events</div>';
        $html_body .= '</div>';
        $html_body .= '<div class="stat-card">';
        $html_body .= '<div class="stat-number">' . $stats['active_users'] . '</div>';
        $html_body .= '<div class="stat-label">Active Users</div>';
        $html_body .= '</div>';
        $html_body .= '</div>';
        $html_body .= '</div>';
        
        // Only show if enabled
        if (empty($section_toggles) || !empty($section_toggles['daily_summary'])) {
            if (!empty($stats['events_by_category'])) {
                $html_body .= '<div class="section">';
                $html_body .= '<h3>📈 Events by Category</h3>';
                $html_body .= '<div class="data-table">';
                $html_body .= '<table>';
                $html_body .= '<thead><tr><th>Category</th><th>Count</th></tr></thead>';
                $html_body .= '<tbody>';
                foreach ($stats['events_by_category'] as $category) {
                    $html_body .= '<tr>';
                    $html_body .= '<td>' . esc_html(ucfirst($category->category)) . '</td>';
                    $html_body .= '<td>' . esc_html($category->count) . '</td>';
                    $html_body .= '</tr>';
                }
                $html_body .= '</tbody></table>';
                $html_body .= '</div></div>';
            }
            if (!empty($stats['events_by_priority'])) {
                $html_body .= '<div class="section">';
                $html_body .= '<h3>⚠️ Events by Priority</h3>';
                $html_body .= '<div class="data-table">';
                $html_body .= '<table>';
                $html_body .= '<thead><tr><th>Priority</th><th>Count</th></tr></thead>';
                $html_body .= '<tbody>';
                foreach ($stats['events_by_priority'] as $priority) {
                    $priority_class = 'priority-' . $priority->priority;
                    $html_body .= '<tr>';
                    $html_body .= '<td><span class="priority-badge ' . $priority_class . '">' . esc_html(ucfirst($priority->priority)) . '</span></td>';
                    $html_body .= '<td>' . esc_html($priority->count) . '</td>';
                    $html_body .= '</tr>';
                }
                $html_body .= '</tbody></table>';
                $html_body .= '</div></div>';
            }
            if (!empty($stats['top_events'])) {
                $html_body .= '<div class="section">';
                $html_body .= '<h3>🔥 Top Events</h3>';
                $html_body .= '<div class="data-table">';
                $html_body .= '<table>';
                $html_body .= '<thead><tr><th>Event</th><th>Count</th></tr></thead>';
                $html_body .= '<tbody>';
                foreach (array_slice($stats['top_events'], 0, 5) as $event) {
                    $html_body .= '<tr>';
                    $html_body .= '<td>' . esc_html($event->event_key) . '</td>';
                    $html_body .= '<td>' . esc_html($event->count) . '</td>';
                    $html_body .= '</tr>';
                }
                $html_body .= '</tbody></table>';
                $html_body .= '</div></div>';
            }
        }
        
        $html_body .= '</div>';
        $html_body .= self::get_email_footer();
        
        return array(
            'subject' => $subject,
            'html_body' => $html_body,
            'plain_body' => $plain_body
        );
    }
    
    /**
     * Generate daily error digest email
     * 
     * @param array $error_data Error events data
     * @return array Email data (subject, html_body, plain_body)
     */
    public static function generate_daily_error_email($error_data, $section_toggles = array()) {
        $date = date('F j, Y', strtotime($error_data['date_range']['start']));
        $subject = "Sentinel Daily Error Report - $date";
        
        // Plain text version
        $plain_body = "Sentinel Daily Error Report\n";
        $plain_body .= "Date: $date\n";
        $plain_body .= "Generated: " . current_time('F j, Y g:i A') . "\n\n";
        
        // Only show if enabled
        if (empty($section_toggles) || !empty($section_toggles['daily_error'])) {
            $plain_body .= "❌ ERROR SUMMARY\n";
            $plain_body .= "Total Errors: " . $error_data['total_errors'] . "\n\n";
            if (!empty($error_data['error_events'])) {
                $plain_body .= "🚨 ERROR DETAILS\n";
                foreach (array_slice($error_data['error_events'], 0, 10) as $error) {
                    $plain_body .= "- " . $error->event_key . " (" . $error->priority . ")\n";
                    $plain_body .= "  Time: " . $error->created_at . "\n";
                    if (!empty($error->data)) {
                        $data = maybe_unserialize($error->data);
                        if (is_array($data) && isset($data['error_message'])) {
                            $plain_body .= "  Message: " . $data['error_message'] . "\n";
                        }
                    }
                    $plain_body .= "\n";
                }
            }
        }
        
        $plain_body .= "View full report: " . admin_url('admin.php?page=sentinel-dashboard') . "\n";
        $plain_body .= "Generated by Sentinel - WordPress Activity Logger";
        
        // HTML version
        $html_body = self::get_email_header($subject);
        $html_body .= '<div class="content">';
        
        // Only show if enabled
        if (empty($section_toggles) || !empty($section_toggles['daily_error'])) {
            $html_body .= '<div class="error-summary">';
            $html_body .= '<h2>❌ Error Summary</h2>';
            $html_body .= '<div class="error-stats">';
            $html_body .= '<div class="stat-card error">';
            $html_body .= '<div class="stat-number">' . $error_data['total_errors'] . '</div>';
            $html_body .= '<div class="stat-label">Total Errors</div>';
            $html_body .= '</div>';
            $html_body .= '</div>';
            $html_body .= '</div>';
            if (!empty($error_data['error_events'])) {
                $html_body .= '<div class="section">';
                $html_body .= '<h3>🚨 Error Details</h3>';
                $html_body .= '<div class="error-list">';
                foreach (array_slice($error_data['error_events'], 0, 10) as $error) {
                    $html_body .= '<div class="error-item">';
                    $html_body .= '<div class="error-header">';
                    $html_body .= '<span class="error-type">' . esc_html($error->event_key) . '</span>';
                    $html_body .= '<span class="priority-badge priority-' . esc_attr($error->priority) . '">' . esc_html(ucfirst($error->priority)) . '</span>';
                    $html_body .= '</div>';
                    $html_body .= '<div class="error-time">' . esc_html($error->created_at) . '</div>';
                    if (!empty($error->data)) {
                        $data = maybe_unserialize($error->data);
                        if (is_array($data) && isset($data['error_message'])) {
                            $html_body .= '<div class="error-message">' . esc_html($data['error_message']) . '</div>';
                        }
                    }
                    $html_body .= '</div>';
                }
                $html_body .= '</div></div>';
            }
        }
        
        $html_body .= '</div>';
        $html_body .= self::get_email_footer();
        
        return array(
            'subject' => $subject,
            'html_body' => $html_body,
            'plain_body' => $plain_body
        );
    }
    
    /**
     * Generate daily user activity digest email
     * 
     * @param array $user_data User activity data
     * @return array Email data (subject, html_body, plain_body)
     */
    public static function generate_daily_user_email($user_data, $section_toggles = array()) {
        $date = date('F j, Y', strtotime($user_data['date_range']['start']));
        $subject = "Sentinel Daily User Activity - $date";
        
        // Plain text version
        $plain_body = "Sentinel Daily User Activity Report\n";
        $plain_body .= "Date: $date\n";
        $plain_body .= "Generated: " . current_time('F j, Y g:i A') . "\n\n";
        
        // Only show if enabled
        if (empty($section_toggles) || !empty($section_toggles['daily_user'])) {
            $plain_body .= "👥 USER ACTIVITY SUMMARY\n";
            $plain_body .= "Total User Events: " . $user_data['total_user_events'] . "\n\n";
            if (!empty($user_data['top_users'])) {
                $plain_body .= "🏆 TOP ACTIVE USERS\n";
                foreach ($user_data['top_users'] as $user) {
                    $plain_body .= "- User ID: " . $user->user_id . ": " . $user->activity_count . " activities\n";
                }
                $plain_body .= "\n";
            }
            if (!empty($user_data['user_events'])) {
                $plain_body .= "📝 RECENT USER EVENTS\n";
                foreach (array_slice($user_data['user_events'], 0, 5) as $event) {
                    $plain_body .= "- User ID: " . $event->user_id . ": " . $event->event_key . " (" . $event->created_at . ")\n";
                }
                $plain_body .= "\n";
            }
        }
        
        $plain_body .= "View full report: " . admin_url('admin.php?page=sentinel-dashboard') . "\n";
        $plain_body .= "Generated by Sentinel - WordPress Activity Logger";
        
        // HTML version
        $html_body = self::get_email_header($subject);
        $html_body .= '<div class="content">';
        
        // Only show if enabled
        if (empty($section_toggles) || !empty($section_toggles['daily_user'])) {
            $html_body .= '<div class="overview-section">';
            $html_body .= '<h2>👥 User Activity Summary</h2>';
            $html_body .= '<div class="stats-grid">';
            $html_body .= '<div class="stat-card">';
            $html_body .= '<div class="stat-number">' . $user_data['total_user_events'] . '</div>';
            $html_body .= '<div class="stat-label">Total User Events</div>';
            $html_body .= '</div>';
            $html_body .= '</div>';
            $html_body .= '</div>';
            if (!empty($user_data['top_users'])) {
                $html_body .= '<div class="section">';
                $html_body .= '<h3>🏆 Top Active Users</h3>';
                $html_body .= '<div class="data-table">';
                $html_body .= '<table>';
                $html_body .= '<thead><tr><th>User</th><th>Activities</th></tr></thead>';
                $html_body .= '<tbody>';
                foreach ($user_data['top_users'] as $user) {
                    $html_body .= '<tr>';
                    $html_body .= '<td>User ID: ' . esc_html($user->user_id) . '</td>';
                    $html_body .= '<td>' . esc_html($user->activity_count) . '</td>';
                    $html_body .= '</tr>';
                }
                $html_body .= '</tbody></table>';
                $html_body .= '</div></div>';
            }
            if (!empty($user_data['user_events'])) {
                $html_body .= '<div class="section">';
                $html_body .= '<h3>📝 Recent User Events</h3>';
                $html_body .= '<div class="data-table">';
                $html_body .= '<table>';
                $html_body .= '<thead><tr><th>User</th><th>Event</th><th>Time</th></tr></thead>';
                $html_body .= '<tbody>';
                foreach (array_slice($user_data['user_events'], 0, 5) as $event) {
                    $html_body .= '<tr>';
                    $html_body .= '<td>User ID: ' . esc_html($event->user_id) . '</td>';
                    $html_body .= '<td>' . esc_html($event->event_key) . '</td>';
                    $html_body .= '<td>' . esc_html($event->created_at) . '</td>';
                    $html_body .= '</tr>';
                }
                $html_body .= '</tbody></table>';
                $html_body .= '</div></div>';
            }
        }
        
        $html_body .= '</div>';
        $html_body .= self::get_email_footer();
        
        return array(
            'subject' => $subject,
            'html_body' => $html_body,
            'plain_body' => $plain_body
        );
    }
    
    /**
     * Generate combined daily digest email with all enabled sections
     * 
     * @param array $combined_data All digest data
     * @param array $section_toggles Which sections are enabled
     * @return array Email data (subject, html_body, plain_body)
     */
    public static function generate_combined_daily_digest_email($combined_data, $section_toggles) {
        $date = date('F j, Y');
        $subject = "Sentinel Daily Digest - $date";
        
        // Plain text version
        $plain_body = "Sentinel Daily Digest Report\n";
        $plain_body .= "Date: $date\n";
        $plain_body .= "Generated: " . current_time('F j, Y g:i A') . "\n\n";
        
        // Add enabled sections to plain text
        if (!empty($section_toggles['daily_summary']) && !empty($combined_data['daily_summary'])) {
            $stats = $combined_data['daily_summary'];
            $plain_body .= "📊 EVENT SUMMARY\n";
            $plain_body .= "Total Events: " . $stats['total_events'] . "\n";
            $plain_body .= "Active Users: " . $stats['active_users'] . "\n\n";
            
            if (!empty($stats['events_by_category'])) {
                $plain_body .= "Events by Category:\n";
                foreach ($stats['events_by_category'] as $category) {
                    $plain_body .= "  " . ucfirst($category->category) . ": " . $category->count . "\n";
                }
                $plain_body .= "\n";
            }
            
            if (!empty($stats['events_by_priority'])) {
                $plain_body .= "Events by Priority:\n";
                foreach ($stats['events_by_priority'] as $priority) {
                    $plain_body .= "  " . ucfirst($priority->priority) . ": " . $priority->count . "\n";
                }
                $plain_body .= "\n";
            }
            
            if (!empty($stats['top_events'])) {
                $plain_body .= "Top Events:\n";
                foreach (array_slice($stats['top_events'], 0, 5) as $event) {
                    $plain_body .= "  " . $event->event_key . ": " . $event->count . "\n";
                }
                $plain_body .= "\n";
            }
        }
        
        if (!empty($section_toggles['daily_error']) && !empty($combined_data['daily_error'])) {
            $error_data = $combined_data['daily_error'];
            $plain_body .= "⚠️ ERROR REPORT\n";
            $plain_body .= "Total Errors: " . $error_data['total_errors'] . "\n\n";
            
            if (!empty($error_data['error_events'])) {
                $plain_body .= "Recent Errors:\n";
                foreach (array_slice($error_data['error_events'], 0, 5) as $error) {
                    $plain_body .= "  " . $error->event_key . " - " . $error->created_at . "\n";
                }
                $plain_body .= "\n";
            }
        }
        
        if (!empty($section_toggles['daily_user']) && !empty($combined_data['daily_user'])) {
            $user_data = $combined_data['daily_user'];
            $plain_body .= "👥 USER ACTIVITY\n";
            $plain_body .= "Total User Events: " . $user_data['total_user_events'] . "\n\n";
            
            if (!empty($user_data['top_users'])) {
                $plain_body .= "Top Active Users:\n";
                foreach (array_slice($user_data['top_users'], 0, 5) as $user) {
                    $username = self::get_username_by_id($user->user_id);
                    $plain_body .= "  User ID: " . $user->user_id . " ($username): " . $user->activity_count . " activities\n";
                }
                $plain_body .= "\n";
            }
            
            if (!empty($user_data['user_events'])) {
                $plain_body .= "Recent User Events:\n";
                foreach (array_slice($user_data['user_events'], 0, 3) as $event) {
                    $username = self::get_username_by_id($event->user_id);
                    $plain_body .= "  User: User ID: " . $event->user_id . " ($username)\n";
                    $plain_body .= "  Event: " . $event->event_key . "\n";
                    $plain_body .= "  Time: " . $event->created_at . "\n\n";
                }
            }
        }
        
        $plain_body .= "This report was generated by Sentinel - WordPress Activity Logger\n";
        $plain_body .= "View full dashboard: " . admin_url('admin.php?page=sentinel-dashboard');
        
        // HTML version
        $html_body = '<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>' . htmlspecialchars($subject) . '</title>
<style>
body { 
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; 
    line-height: 1.6; 
    color: #374151; 
    margin: 0; 
    padding: 0; 
    background-color: #f9fafb; 
} 
.container { 
    max-width: 700px; 
    margin: 0 auto; 
    background-color: #ffffff; 
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1); 
} 
.header { 
    background: #fff; 
    color: #23282d; 
    padding: 32px 24px 20px 24px; 
    text-align: center; 
    border-bottom: 1px solid #e5e7eb; 
} 
.header img { 
    max-height: 64px; 
    width: auto; 
    margin-bottom: 16px; 
    display: block; 
    margin-left: auto; 
    margin-right: auto; 
} 
.header h1 { 
    margin: 0; 
    font-size: 28px; 
    font-weight: 700; 
    letter-spacing: 0.5px; 
    color: #23282d; 
} 
.header p { 
    margin: 12px 0 0 0; 
    opacity: 0.85; 
    font-size: 14px; 
    color: #6b7280; 
} 
.content { 
    padding: 40px 30px; 
} 
.section { 
    margin-bottom: 40px; 
    background-color: #f8fafc; 
    border: 1px solid #e5e7eb; 
    border-radius: 12px; 
    overflow: hidden; 
} 
.section-header { 
    background-color: #ffffff; 
    padding: 20px 24px; 
    border-bottom: 1px solid #e5e7eb; 
    text-align: center; 
} 
.section-title { 
    margin: 0; 
    font-size: 20px; 
    font-weight: 600; 
    color: #111827; 
    display: flex; 
    align-items: center; 
    justify-content: center; 
    gap: 8px; 
} 
.section-content { 
    padding: 24px; 
} 
.overview-grid { 
    display: grid; 
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); 
    gap: 20px; 
    margin-bottom: 30px; 
} 
.overview-card { 
    background-color: #ffffff; 
    padding: 20px; 
    border-radius: 8px; 
    border: 1px solid #e5e7eb; 
    text-align: center; 
} 
.overview-number { 
    font-size: 32px; 
    font-weight: 700; 
    color: #2563eb; 
    margin-bottom: 8px; 
} 
.overview-label { 
    font-size: 14px; 
    color: #6b7280; 
    text-transform: uppercase; 
    letter-spacing: 0.5px; 
    font-weight: 600; 
} 
.data-table { 
    width: 100%; 
    border-collapse: collapse; 
    margin-top: 20px; 
    background-color: #ffffff; 
    border-radius: 8px; 
    overflow: hidden; 
    border: 1px solid #e5e7eb; 
    table-layout: fixed; 
} 
.data-table th { 
    background-color: #f8fafc; 
    padding: 12px 16px; 
    text-align: left; 
    font-weight: 600; 
    color: #374151; 
    border-bottom: 1px solid #e5e7eb; 
} 
.data-table td { 
    padding: 12px 16px; 
    border-bottom: 1px solid #f3f4f6; 
    color: #111827; 
} 
.data-table tr:last-child td { 
    border-bottom: none; 
} 
/* Two-column tables (Category, Priority, User Activities) */
.data-table th:first-child:nth-last-child(2),
.data-table td:first-child:nth-last-child(2) { 
    width: 70%; 
} 
.data-table th:last-child:nth-child(2),
.data-table td:last-child:nth-child(2) { 
    width: 30%; 
    text-align: center; 
    font-weight: 600; 
}
/* Three-column tables (Recent User Events) */
.data-table th:nth-child(1):nth-last-child(3),
.data-table td:nth-child(1):nth-last-child(3) { 
    width: 40%; 
} 
.data-table th:nth-child(2):nth-last-child(2),
.data-table td:nth-child(2):nth-last-child(2) { 
    width: 35%; 
} 
.data-table th:nth-child(3):nth-last-child(1),
.data-table td:nth-child(3):nth-last-child(1) { 
    width: 25%; 
}
.priority-badge { 
    display: inline-block; 
    padding: 4px 12px; 
    border-radius: 20px; 
    font-size: 12px; 
    font-weight: 600; 
    text-transform: uppercase; 
    letter-spacing: 0.5px; 
} 
.priority-critical { 
    background-color: #dc2626; 
    color: white; 
} 
.priority-high { 
    background-color: #ea580c; 
    color: white; 
} 
.priority-medium { 
    background-color: #ca8a04; 
    color: white; 
} 
.priority-low { 
    background-color: #2563eb; 
    color: white; 
} 
.user-info { 
    display: flex; 
    align-items: center; 
    gap: 8px; 
} 
.user-id { 
    font-weight: 600; 
    color: #111827; 
} 
.username { 
    color: #6b7280; 
    font-size: 14px; 
} 
.footer { 
    background-color: #f8fafc; 
    border-top: 1px solid #e5e7eb; 
    padding: 24px; 
    text-align: center; 
} 
.footer p { 
    margin: 0 0 8px 0; 
    font-size: 14px; 
    color: #6b7280; 
} 
.footer a { 
    color: #3b82f6; 
    text-decoration: none; 
} 
.footer a:hover { 
    text-decoration: underline; 
} 
@media (max-width: 600px) { 
    .overview-grid { 
        grid-template-columns: 1fr; 
        gap: 16px; 
    } 
    .content { 
        padding: 24px 20px; 
    } 
    .header { 
        padding: 24px 16px 16px 16px; 
    } 
    .section-content { 
        padding: 20px; 
    } 
    .data-table { 
        font-size: 14px; 
    } 
    .data-table th, 
    .data-table td { 
        padding: 10px 12px; 
    } 
} 
</style>
</head>
<body>
<div class="container">
<div class="header">
<img src="https://www.justpsntrophies.com/wp-content/themes/jpsnt/img/Sentinel.png" alt="Sentinel Logo" />
<h1>' . htmlspecialchars($subject) . '</h1>
<p>WordPress Activity Monitoring System</p>
</div>
<div class="content">';
        
        // Event Summary Section
        if (!empty($section_toggles['daily_summary']) && !empty($combined_data['daily_summary'])) {
            $stats = $combined_data['daily_summary'];
            $html_body .= '
<div class="section">
<div class="section-header">
<h2 class="section-title">📊 Event Summary</h2>
</div>
<div class="section-content">
<div class="overview-grid">
<div class="overview-card">
<div class="overview-number">' . $stats['total_events'] . '</div>
<div class="overview-label">Total Events</div>
</div>
<div class="overview-card">
<div class="overview-number">' . $stats['active_users'] . '</div>
<div class="overview-label">Active Users</div>
</div>
</div>';
            
            if (!empty($stats['events_by_category'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>Category</th>
<th>Count</th>
</tr>
</thead>
<tbody>';
                foreach ($stats['events_by_category'] as $category) {
                    $html_body .= '
<tr>
<td>' . htmlspecialchars(ucfirst($category->category)) . '</td>
<td>' . $category->count . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            if (!empty($stats['events_by_priority'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>Priority</th>
<th>Count</th>
</tr>
</thead>
<tbody>';
                foreach ($stats['events_by_priority'] as $priority) {
                    $priority_class = 'priority-' . strtolower($priority->priority);
                    $html_body .= '
<tr>
<td><span class="priority-badge ' . $priority_class . '">' . htmlspecialchars(strtoupper($priority->priority)) . '</span></td>
<td>' . $priority->count . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            $html_body .= '
</div>
</div>';
        }
        
        // Error Report Section
        if (!empty($section_toggles['daily_error']) && !empty($combined_data['daily_error'])) {
            $error_data = $combined_data['daily_error'];
            $html_body .= '
<div class="section">
<div class="section-header">
<h2 class="section-title">⚠️ Error Report</h2>
</div>
<div class="section-content">
<div class="overview-grid">
<div class="overview-card">
<div class="overview-number">' . $error_data['total_errors'] . '</div>
<div class="overview-label">Total Errors</div>
</div>
</div>';
            
            if (!empty($error_data['error_events'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>Error Type</th>
<th>Time</th>
</tr>
</thead>
<tbody>';
                foreach (array_slice($error_data['error_events'], 0, 5) as $error) {
                    $html_body .= '
<tr>
<td>' . htmlspecialchars($error->event_key) . '</td>
<td>' . htmlspecialchars($error->created_at) . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            $html_body .= '
</div>
</div>';
        }
        
        // User Activity Section
        if (!empty($section_toggles['daily_user']) && !empty($combined_data['daily_user'])) {
            $user_data = $combined_data['daily_user'];
            $html_body .= '
<div class="section">
<div class="section-header">
<h2 class="section-title">👥 User Activity</h2>
</div>
<div class="section-content">
<div class="overview-grid">
<div class="overview-card">
<div class="overview-number">' . $user_data['total_user_events'] . '</div>
<div class="overview-label">Total User Events</div>
</div>
</div>';
            
            if (!empty($user_data['top_users'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>User</th>
<th>Activities</th>
</tr>
</thead>
<tbody>';
                foreach (array_slice($user_data['top_users'], 0, 5) as $user) {
                    $username = self::get_username_by_id($user->user_id);
                    $html_body .= '
<tr>
<td>
<div class="user-info">
<span class="user-id">User ID: ' . $user->user_id . '</span>
<span class="username">(' . htmlspecialchars($username) . ')</span>
</div>
</td>
<td>' . $user->activity_count . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            if (!empty($user_data['user_events'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>User</th>
<th>Event</th>
<th>Time</th>
</tr>
</thead>
<tbody>';
                foreach (array_slice($user_data['user_events'], 0, 3) as $event) {
                    $username = self::get_username_by_id($event->user_id);
                    $html_body .= '
<tr>
<td>
<div class="user-info">
<span class="user-id">User ID: ' . $event->user_id . '</span>
<span class="username">(' . htmlspecialchars($username) . ')</span>
</div>
</td>
<td>' . htmlspecialchars($event->event_key) . '</td>
<td>' . htmlspecialchars($event->created_at) . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            $html_body .= '
</div>
</div>';
        }
        
        $html_body .= '
</div>
<div class="footer">
<p>This report was generated by <a href="http://www.getsentinel.dev/">Sentinel</a> - WordPress Activity Logger</p>
<p><a href="' . admin_url('admin.php?page=sentinel-dashboard') . '">View full dashboard</a></p>
</div>
</div>
</body>
</html>';
        
        return array(
            'subject' => $subject,
            'html_body' => $html_body,
            'plain_body' => $plain_body
        );
    }
    
    /**
     * Generate combined weekly digest email with all enabled sections
     * 
     * @param array $combined_data All digest data
     * @param array $section_toggles Which sections are enabled
     * @return array Email data (subject, html_body, plain_body)
     */
    public static function generate_combined_weekly_digest_email($combined_data, $section_toggles) {
        $date = date('F j, Y');
        $subject = "Sentinel Weekly Digest - $date";
        
        // Plain text version
        $plain_body = "Sentinel Weekly Digest Report\n";
        $plain_body .= "Date: $date\n";
        $plain_body .= "Generated: " . current_time('F j, Y g:i A') . "\n\n";
        
        // Add enabled sections to plain text
        if (!empty($section_toggles['weekly_health']) && !empty($combined_data['weekly_health'])) {
            $plain_body .= "🏥 SYSTEM HEALTH REPORT\n";
            $plain_body .= "Total Events: " . (isset($combined_data['weekly_health']['total_events']) ? $combined_data['weekly_health']['total_events'] : 'N/A') . "\n";
            $plain_body .= "System Uptime: " . (isset($combined_data['weekly_health']['uptime_percentage']) ? $combined_data['weekly_health']['uptime_percentage'] : 'N/A') . "%\n\n";
            
            if (!empty($combined_data['weekly_health']['health_metrics'])) {
                $plain_body .= "Health Metrics:\n";
                foreach ($combined_data['weekly_health']['health_metrics'] as $metric => $value) {
                    $plain_body .= "  $metric: $value\n";
                }
                $plain_body .= "\n";
            }
        }
        
        if (!empty($section_toggles['weekly_performance']) && !empty($combined_data['weekly_performance'])) {
            $plain_body .= "⚡ PERFORMANCE METRICS\n";
            $plain_body .= "Average Response Time: " . (isset($combined_data['weekly_performance']['avg_response_time']) ? $combined_data['weekly_performance']['avg_response_time'] : 'N/A') . "ms\n";
            $plain_body .= "Peak Memory Usage: " . (isset($combined_data['weekly_performance']['peak_memory']) ? $combined_data['weekly_performance']['peak_memory'] : 'N/A') . "\n\n";
            
            if (!empty($combined_data['weekly_performance']['performance_trends'])) {
                $plain_body .= "Performance Trends:\n";
                foreach ($combined_data['weekly_performance']['performance_trends'] as $trend) {
                    $plain_body .= "  " . $trend['metric'] . ": " . $trend['value'] . "\n";
                }
                $plain_body .= "\n";
            }
        }
        
        if (!empty($section_toggles['weekly_security']) && !empty($combined_data['weekly_security'])) {
            $plain_body .= "🔒 SECURITY SUMMARY\n";
            $plain_body .= "Security Events: " . (isset($combined_data['weekly_security']['security_events']) ? $combined_data['weekly_security']['security_events'] : 'N/A') . "\n";
            $plain_body .= "Failed Logins: " . (isset($combined_data['weekly_security']['failed_logins']) ? $combined_data['weekly_security']['failed_logins'] : 'N/A') . "\n\n";
            
            if (!empty($combined_data['weekly_security']['security_incidents'])) {
                $plain_body .= "Security Incidents:\n";
                foreach (array_slice($combined_data['weekly_security']['security_incidents'], 0, 5) as $incident) {
                    $plain_body .= "  " . $incident['type'] . " - " . $incident['date'] . "\n";
                }
                $plain_body .= "\n";
            }
        }
        
        $plain_body .= "This report was generated by Sentinel - WordPress Activity Logger\n";
        $plain_body .= "View full dashboard: " . admin_url('admin.php?page=sentinel-dashboard');
        
        // HTML version
        $html_body = '<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>' . htmlspecialchars($subject) . '</title>
<style>
body { 
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; 
    line-height: 1.6; 
    color: #374151; 
    margin: 0; 
    padding: 0; 
    background-color: #f9fafb; 
} 
.container { 
    max-width: 700px; 
    margin: 0 auto; 
    background-color: #ffffff; 
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1); 
} 
.header { 
    background: #fff; 
    color: #23282d; 
    padding: 32px 24px 20px 24px; 
    text-align: center; 
    border-bottom: 1px solid #e5e7eb; 
} 
.header img { 
    max-height: 64px; 
    width: auto; 
    margin-bottom: 16px; 
    display: block; 
    margin-left: auto; 
    margin-right: auto; 
} 
.header h1 { 
    margin: 0; 
    font-size: 28px; 
    font-weight: 700; 
    letter-spacing: 0.5px; 
    color: #23282d; 
} 
.header p { 
    margin: 12px 0 0 0; 
    opacity: 0.85; 
    font-size: 14px; 
    color: #6b7280; 
} 
.content { 
    padding: 40px 30px; 
} 
.section { 
    margin-bottom: 40px; 
    background-color: #f8fafc; 
    border: 1px solid #e5e7eb; 
    border-radius: 12px; 
    overflow: hidden; 
} 
.section-header { 
    background-color: #ffffff; 
    padding: 20px 24px; 
    border-bottom: 1px solid #e5e7eb; 
    text-align: center; 
} 
.section-title { 
    margin: 0; 
    font-size: 20px; 
    font-weight: 600; 
    color: #111827; 
    display: flex; 
    align-items: center; 
    justify-content: center; 
    gap: 8px; 
} 
.section-content { 
    padding: 24px; 
} 
.overview-grid { 
    display: grid; 
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); 
    gap: 20px; 
    margin-bottom: 30px; 
} 
.overview-card { 
    background-color: #ffffff; 
    padding: 20px; 
    border-radius: 8px; 
    border: 1px solid #e5e7eb; 
    text-align: center; 
} 
.overview-number { 
    font-size: 32px; 
    font-weight: 700; 
    color: #2563eb; 
    margin-bottom: 8px; 
} 
.overview-label { 
    font-size: 14px; 
    color: #6b7280; 
    text-transform: uppercase; 
    letter-spacing: 0.5px; 
    font-weight: 600; 
} 
.data-table { 
    width: 100%; 
    border-collapse: collapse; 
    margin-top: 20px; 
    background-color: #ffffff; 
    border-radius: 8px; 
    overflow: hidden; 
    border: 1px solid #e5e7eb; 
    table-layout: fixed; 
} 
.data-table th { 
    background-color: #f8fafc; 
    padding: 12px 16px; 
    text-align: left; 
    font-weight: 600; 
    color: #374151; 
    border-bottom: 1px solid #e5e7eb; 
} 
.data-table td { 
    padding: 12px 16px; 
    border-bottom: 1px solid #f3f4f6; 
    color: #111827; 
} 
.data-table tr:last-child td { 
    border-bottom: none; 
} 
.data-table th:first-child,
.data-table td:first-child { 
    width: 70%; 
} 
.data-table th:last-child,
.data-table td:last-child { 
    width: 30%; 
    text-align: center; 
    font-weight: 600; 
}
.metric-grid { 
    display: grid; 
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); 
    gap: 16px; 
    margin-top: 20px; 
} 
.metric-item { 
    background-color: #ffffff; 
    padding: 16px; 
    border-radius: 8px; 
    border: 1px solid #e5e7eb; 
    text-align: center; 
} 
.metric-value { 
    font-size: 24px; 
    font-weight: 700; 
    color: #059669; 
    margin-bottom: 4px; 
} 
.metric-label { 
    font-size: 12px; 
    color: #6b7280; 
    text-transform: uppercase; 
    letter-spacing: 0.5px; 
    font-weight: 600; 
} 
.footer { 
    background-color: #f8fafc; 
    border-top: 1px solid #e5e7eb; 
    padding: 24px; 
    text-align: center; 
} 
.footer p { 
    margin: 0 0 8px 0; 
    font-size: 14px; 
    color: #6b7280; 
} 
.footer a { 
    color: #3b82f6; 
    text-decoration: none; 
} 
.footer a:hover { 
    text-decoration: underline; 
} 
@media (max-width: 600px) { 
    .overview-grid { 
        grid-template-columns: 1fr; 
        gap: 16px; 
    } 
    .metric-grid { 
        grid-template-columns: 1fr; 
        gap: 12px; 
    } 
    .content { 
        padding: 24px 20px; 
    } 
    .header { 
        padding: 24px 16px 16px 16px; 
    } 
    .section-content { 
        padding: 20px; 
    } 
    .data-table { 
        font-size: 14px; 
    } 
    .data-table th, 
    .data-table td { 
        padding: 10px 12px; 
    } 
} 
</style>
</head>
<body>
<div class="container">
<div class="header">
<img src="https://www.justpsntrophies.com/wp-content/themes/jpsnt/img/Sentinel.png" alt="Sentinel Logo" />
<h1>' . htmlspecialchars($subject) . '</h1>
<p>WordPress Activity Monitoring System</p>
</div>
<div class="content">';
        
        // System Health Section
        if (!empty($section_toggles['weekly_health']) && !empty($combined_data['weekly_health'])) {
            $html_body .= '
<div class="section">
<div class="section-header">
<h2 class="section-title">🏥 System Health Report</h2>
</div>
<div class="section-content">
<div class="overview-grid">
<div class="overview-card">
<div class="overview-number">' . (isset($combined_data['weekly_health']['total_events']) ? $combined_data['weekly_health']['total_events'] : 'N/A') . '</div>
<div class="overview-label">Total Events</div>
</div>
<div class="overview-card">
<div class="overview-number">' . (isset($combined_data['weekly_health']['uptime_percentage']) ? $combined_data['weekly_health']['uptime_percentage'] : 'N/A') . '%</div>
<div class="overview-label">System Uptime</div>
</div>
</div>';
            
            if (!empty($combined_data['weekly_health']['health_metrics'])) {
                $html_body .= '
<div class="metric-grid">';
                foreach ($combined_data['weekly_health']['health_metrics'] as $metric => $value) {
                    $html_body .= '
<div class="metric-item">
<div class="metric-value">' . htmlspecialchars($value) . '</div>
<div class="metric-label">' . htmlspecialchars($metric) . '</div>
</div>';
                }
                $html_body .= '
</div>';
            }
            
            $html_body .= '
</div>
</div>';
        }
        
        // Performance Metrics Section
        if (!empty($section_toggles['weekly_performance']) && !empty($combined_data['weekly_performance'])) {
            $html_body .= '
<div class="section">
<div class="section-header">
<h2 class="section-title">⚡ Performance Metrics</h2>
</div>
<div class="section-content">
<div class="overview-grid">
<div class="overview-card">
<div class="overview-number">' . (isset($combined_data['weekly_performance']['avg_response_time']) ? $combined_data['weekly_performance']['avg_response_time'] : 'N/A') . 'ms</div>
<div class="overview-label">Avg Response Time</div>
</div>
<div class="overview-card">
<div class="overview-number">' . (isset($combined_data['weekly_performance']['peak_memory']) ? $combined_data['weekly_performance']['peak_memory'] : 'N/A') . '</div>
<div class="overview-label">Peak Memory Usage</div>
</div>
</div>';
            
            if (!empty($combined_data['weekly_performance']['performance_trends'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>Metric</th>
<th>Value</th>
</tr>
</thead>
<tbody>';
                foreach ($combined_data['weekly_performance']['performance_trends'] as $trend) {
                    $html_body .= '
<tr>
<td>' . htmlspecialchars($trend['metric']) . '</td>
<td>' . htmlspecialchars($trend['value']) . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            $html_body .= '
</div>
</div>';
        }
        
        // Security Summary Section
        if (!empty($section_toggles['weekly_security']) && !empty($combined_data['weekly_security'])) {
            $html_body .= '
<div class="section">
<div class="section-header">
<h2 class="section-title">🔒 Security Summary</h2>
</div>
<div class="section-content">
<div class="overview-grid">
<div class="overview-card">
<div class="overview-number">' . (isset($combined_data['weekly_security']['security_events']) ? $combined_data['weekly_security']['security_events'] : 'N/A') . '</div>
<div class="overview-label">Security Events</div>
</div>
<div class="overview-card">
<div class="overview-number">' . (isset($combined_data['weekly_security']['failed_logins']) ? $combined_data['weekly_security']['failed_logins'] : 'N/A') . '</div>
<div class="overview-label">Failed Logins</div>
</div>
</div>';
            
            if (!empty($combined_data['weekly_security']['security_incidents'])) {
                $html_body .= '
<table class="data-table">
<thead>
<tr>
<th>Incident Type</th>
<th>Date</th>
</tr>
</thead>
<tbody>';
                foreach (array_slice($combined_data['weekly_security']['security_incidents'], 0, 5) as $incident) {
                    $html_body .= '
<tr>
<td>' . htmlspecialchars($incident['type']) . '</td>
<td>' . htmlspecialchars($incident['date']) . '</td>
</tr>';
                }
                $html_body .= '
</tbody>
</table>';
            }
            
            $html_body .= '
</div>
</div>';
        }
        
        $html_body .= '
</div>
<div class="footer">
<p>This report was generated by <a href="http://www.getsentinel.dev/">Sentinel</a> - WordPress Activity Logger</p>
<p><a href="' . admin_url('admin.php?page=sentinel-dashboard') . '">View full dashboard</a></p>
</div>
</div>
</body>
</html>';
        
        return array(
            'subject' => $subject,
            'html_body' => $html_body,
            'plain_body' => $plain_body
        );
    }
    
    /**
     * Generate weekly health digest email
     * 
     * @param array $stats Weekly statistics data
     * @return array Email data (subject, html_body, plain_body)
     */
    public static function generate_weekly_health_email($stats) {
        $start_date = date('F j', strtotime($stats['date_range']['start']));
        $end_date = date('F j, Y', strtotime($stats['date_range']['end']));
        $subject = "Sentinel Weekly Health Report - $start_date to $end_date";
        
        // Plain text version
        $plain_body = "Sentinel Weekly Health Report\n";
        $plain_body .= "Period: $start_date to $end_date\n";
        $plain_body .= "Generated: " . current_time('F j, Y g:i A') . "\n\n";
        
        $plain_body .= "📊 WEEKLY OVERVIEW\n";
        $plain_body .= "Total Events: " . $stats['total_events'] . "\n";
        $plain_body .= "Active Users: " . $stats['active_users'] . "\n\n";
        
        if (!empty($stats['daily_breakdown'])) {
            $plain_body .= "📅 DAILY BREAKDOWN\n";
            foreach ($stats['daily_breakdown'] as $day) {
                $plain_body .= "- " . date('M j', strtotime($day['date'])) . ": " . $day['count'] . " events\n";
            }
            $plain_body .= "\n";
        }
        
        if (!empty($stats['events_by_category'])) {
            $plain_body .= "📈 EVENTS BY CATEGORY\n";
            foreach ($stats['events_by_category'] as $category) {
                $plain_body .= "- " . ucfirst($category->category) . ": " . $category->count . "\n";
            }
            $plain_body .= "\n";
        }
        
        $plain_body .= "View full report: " . admin_url('admin.php?page=sentinel-dashboard') . "\n";
        $plain_body .= "Generated by Sentinel - WordPress Activity Logger";
        
        // HTML version
        $html_body = self::get_email_header($subject);
        $html_body .= '<div class="content">';
        
        // Weekly overview
        $html_body .= '<div class="overview-section">';
        $html_body .= '<h2>📊 Weekly Overview</h2>';
        $html_body .= '<div class="stats-grid">';
        $html_body .= '<div class="stat-card">';
        $html_body .= '<div class="stat-number">' . $stats['total_events'] . '</div>';
        $html_body .= '<div class="stat-label">Total Events</div>';
        $html_body .= '</div>';
        $html_body .= '<div class="stat-card">';
        $html_body .= '<div class="stat-number">' . $stats['active_users'] . '</div>';
        $html_body .= '<div class="stat-label">Active Users</div>';
        $html_body .= '</div>';
        $html_body .= '</div>';
        $html_body .= '</div>';
        
        // Daily breakdown
        if (!empty($stats['daily_breakdown'])) {
            $html_body .= '<div class="section">';
            $html_body .= '<h3>📅 Daily Breakdown</h3>';
            $html_body .= '<div class="daily-chart">';
            foreach ($stats['daily_breakdown'] as $day) {
                $date = date('M j', strtotime($day['date']));
                $percentage = $stats['total_events'] > 0 ? ($day['count'] / $stats['total_events']) * 100 : 0;
                $html_body .= '<div class="day-bar">';
                $html_body .= '<div class="day-label">' . esc_html($date) . '</div>';
                $html_body .= '<div class="bar-container">';
                $html_body .= '<div class="bar" style="width: ' . $percentage . '%"></div>';
                $html_body .= '</div>';
                $html_body .= '<div class="day-count">' . esc_html($day['count']) . '</div>';
                $html_body .= '</div>';
            }
            $html_body .= '</div></div>';
        }
        
        // Events by category
        if (!empty($stats['events_by_category'])) {
            $html_body .= '<div class="section">';
            $html_body .= '<h3>📈 Events by Category</h3>';
            $html_body .= '<div class="data-table">';
            $html_body .= '<table>';
            $html_body .= '<thead><tr><th>Category</th><th>Count</th></tr></thead>';
            $html_body .= '<tbody>';
            foreach ($stats['events_by_category'] as $category) {
                $html_body .= '<tr>';
                $html_body .= '<td>' . esc_html(ucfirst($category->category)) . '</td>';
                $html_body .= '<td>' . esc_html($category->count) . '</td>';
                $html_body .= '</tr>';
            }
            $html_body .= '</tbody></table>';
            $html_body .= '</div></div>';
        }
        
        $html_body .= '</div>';
        $html_body .= self::get_email_footer();
        
        return array(
            'subject' => $subject,
            'html_body' => $html_body,
            'plain_body' => $plain_body
        );
    }
    
    /**
     * Get email header with styling
     * 
     * @param string $title Email title
     * @return string HTML header
     */
    private static function get_email_header($title) {
        return '<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>' . esc_html($title) . '</title>
<style>
body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; line-height: 1.6; color: #374151; margin: 0; padding: 0; background-color: #f9fafb; }
.container { max-width: 600px; margin: 0 auto; background-color: #ffffff; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1); }
.header { background: #fff; color: #23282d; padding: 24px 24px 10px 24px; text-align: center; border-bottom: 1px solid #e5e7eb; }
.header img { max-height: 56px; width: auto; margin-bottom: 10px; display: block; margin-left: auto; margin-right: auto; }
.header h1 { margin: 0; font-size: 22px; font-weight: 700; letter-spacing: 0.5px; color: #23282d; }
.header p { margin: 8px 0 0 0; opacity: 0.85; font-size: 13px; color: #6b7280; }
.content { padding: 30px; }
.overview-section { margin-bottom: 30px; }
.overview-section h2 { margin: 0 0 20px 0; color: #111827; font-size: 18px; }
.stats-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; }
.stat-card { background: #f8fafc; padding: 20px; border-radius: 8px; text-align: center; border: 1px solid #e5e7eb; }
.stat-card.error { background: #fef2f2; border-color: #fecaca; }
.stat-number { font-size: 32px; font-weight: 700; color: #111827; margin-bottom: 5px; }
.stat-card.error .stat-number { color: #dc2626; }
.stat-label { font-size: 14px; color: #6b7280; text-transform: uppercase; letter-spacing: 0.5px; }
.section { margin-bottom: 30px; }
.section h3 { margin: 0 0 15px 0; color: #111827; font-size: 16px; }
.data-table { background: #f8fafc; border: 1px solid #e5e7eb; border-radius: 8px; overflow: hidden; }
.data-table table { width: 100%; border-collapse: collapse; }
.data-table th { background: #f1f5f9; padding: 12px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; color: #64748b; }
.data-table td { padding: 12px; border-top: 1px solid #e5e7eb; }
.priority-badge { display: inline-block; padding: 4px 8px; border-radius: 12px; font-size: 11px; font-weight: 600; text-transform: uppercase; }
.priority-critical { background: #fef2f2; color: #dc2626; }
.priority-error { background: #fef2f2; color: #dc2626; }
.priority-warning { background: #fffbeb; color: #d97706; }
.priority-info { background: #eff6ff; color: #2563eb; }
.error-summary { margin-bottom: 30px; }
.error-stats { display: flex; gap: 20px; }
.error-list { background: #f8fafc; border: 1px solid #e5e7eb; border-radius: 8px; }
.error-item { padding: 15px; border-bottom: 1px solid #e5e7eb; }
.error-item:last-child { border-bottom: none; }
.error-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px; }
.error-type { font-weight: 600; color: #111827; }
.error-time { font-size: 12px; color: #6b7280; margin-bottom: 5px; }
.error-message { font-size: 13px; color: #374151; background: #fff; padding: 8px; border-radius: 4px; border: 1px solid #d1d5db; }
.daily-chart { background: #f8fafc; border: 1px solid #e5e7eb; border-radius: 8px; padding: 20px; }
.day-bar { display: flex; align-items: center; margin-bottom: 10px; }
.day-label { width: 60px; font-size: 12px; font-weight: 600; color: #374151; }
.bar-container { flex: 1; height: 20px; background: #e5e7eb; border-radius: 10px; margin: 0 15px; overflow: hidden; }
.bar { height: 100%; background: linear-gradient(90deg, #3b82f6, #1d4ed8); border-radius: 10px; }
.day-count { width: 40px; text-align: right; font-size: 12px; font-weight: 600; color: #374151; }
.footer { background-color: #f8fafc; border-top: 1px solid #e5e7eb; padding: 20px; text-align: center; }
.footer p { margin: 0; font-size: 12px; color: #6b7280; }
.footer a { color: #3b82f6; text-decoration: none; }
@media (max-width: 600px) { .stats-grid { grid-template-columns: 1fr; } .content { padding: 20px; } .header { padding: 16px 8px 8px 8px; } }
</style>
</head>
<body>
<div class="container">
<div class="header">
<img src="https://www.justpsntrophies.com/wp-content/themes/jpsnt/img/Sentinel.png" alt="Sentinel Logo" />
<h1>' . esc_html($title) . '</h1>
<p>WordPress Activity Monitoring System</p>
</div>';
    }
    
    /**
     * Get email footer
     * 
     * @return string HTML footer
     */
    private static function get_email_footer() {
        return '<div class="footer">
<p>This report was generated by <a href="http://www.getsentinel.dev/">Sentinel</a> - WordPress Activity Logger</p>
<p>View full dashboard: <a href="' . admin_url('admin.php?page=sentinel-dashboard') . '">' . admin_url('admin.php?page=sentinel-dashboard') . '</a></p>
</div>
</div>
</body>
</html>';
    }

    /**
     * Helper to get username by user ID.
     *
     * @param int $user_id The user ID.
     * @return string The username.
     */
    private static function get_username_by_id($user_id) {
        $user = get_userdata($user_id);
        return $user ? $user->user_login : 'Unknown User';
    }
} 