<?php
/**
 * Sentinel Admin Class
 * 
 * Handles admin interface and settings.
 * This class creates the admin menu and handles all admin-related functionality.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Sentinel_Admin {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Constructor - private for singleton pattern
     */
    private function __construct() {
        // Constructor is called when we get the instance
    }
    
    /**
     * Get singleton instance
     * 
     * @return Sentinel_Admin
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        
        return self::$instance;
    }
    
    /**
     * Initialize admin functionality
     * This is called from the core class
     */
    public function init() {
        // Add admin menu items
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Add admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Add admin notices
        add_action('admin_notices', array($this, 'admin_notices'));
    }
    
    /**
     * Add admin menu items to WordPress admin
     * This creates the menu items you see in the WordPress admin sidebar
     */
    public function add_admin_menu() {
        // Add main menu item
        add_menu_page(
            'Sentinel Activity Logger',           // Page title (shown in browser tab)
            'Sentinel',                           // Menu title (shown in sidebar)
            'manage_options',                     // Capability required (admin only)
            'sentinel',                           // Menu slug (used in URLs)
            array($this, 'admin_page'),           // Callback function to display page
            'dashicons-shield-alt',               // Icon (WordPress dashicons)
            30                                    // Position in menu (lower = higher up)
        );
        
        // Add submenu items
        add_submenu_page(
            'sentinel',                           // Parent slug
            'Activity Logs',                      // Page title
            'Activity Logs',                      // Menu title
            'manage_options',                     // Capability
            'sentinel',                           // Menu slug (same as parent for first submenu)
            array($this, 'admin_page')            // Callback
        );
        
        add_submenu_page(
            'sentinel',                           // Parent slug
            'Settings',                           // Page title
            'Settings',                           // Menu title
            'manage_options',                     // Capability
            'sentinel-settings',                  // Menu slug
            array($this, 'settings_page')         // Callback
        );
        
        add_submenu_page(
            'sentinel',                           // Parent slug
            'Event Registry',                     // Page title
            'Event Registry',                     // Menu title
            'manage_options',                     // Capability
            'sentinel-events',                    // Menu slug
            array($this, 'events_page')           // Callback
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     * This loads CSS and JavaScript files for the admin interface
     * 
     * @param string $hook The current admin page hook
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our plugin pages to avoid conflicts
        if (strpos($hook, 'sentinel') !== false) {
            // Enqueue CSS
            wp_enqueue_style(
                'sentinel-admin',                  // Handle (unique identifier)
                SENTINEL_PLUGIN_URL . 'admin/css/sentinel-admin.css',  // File path
                array(),                          // Dependencies (none for now)
                SENTINEL_VERSION                  // Version (for cache busting)
            );
            
            // Enqueue JavaScript
            wp_enqueue_script(
                'sentinel-admin',                  // Handle
                SENTINEL_PLUGIN_URL . 'admin/js/sentinel-admin.js',    // File path
                array('jquery'),                  // Dependencies (jQuery)
                SENTINEL_VERSION,                 // Version
                true                              // Load in footer
            );
            
            // Localize script (pass PHP data to JavaScript)
            wp_localize_script('sentinel-admin', 'sentinel_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('sentinel_nonce'),
                'strings' => array(
                    'loading' => 'Loading...',
                    'error' => 'An error occurred',
                    'success' => 'Operation completed successfully'
                )
            ));
        }
    }
    
    /**
     * Display admin notices
     * This shows success/error messages in the admin area
     */
    public function admin_notices() {
        // Check if we have any notices to show
        if (isset($_GET['page']) && strpos($_GET['page'], 'sentinel') !== false) {
            // Example notice - we'll expand this later
            if (isset($_GET['sentinel_notice'])) {
                $notice_type = sanitize_text_field($_GET['sentinel_notice']);
                $message = '';
                
                switch ($notice_type) {
                    case 'activated':
                        $message = 'Sentinel plugin activated successfully!';
                        $class = 'notice-success';
                        break;
                    case 'error':
                        $message = 'An error occurred with Sentinel.';
                        $class = 'notice-error';
                        break;
                    default:
                        return;
                }
                
                if ($message) {
                    echo '<div class="notice ' . esc_attr($class) . ' is-dismissible">';
                    echo '<p>' . esc_html($message) . '</p>';
                    echo '</div>';
                }
            }
        }
    }
    
    /**
     * Main admin page callback
     * This function is called when someone visits the main Sentinel page
     */
    public function admin_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        // Include the admin page view
        include SENTINEL_PLUGIN_DIR . 'admin/views/admin-page.php';
    }
    
    /**
     * Settings page callback
     * This function is called when someone visits the settings page
     */
    public function settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        // Include the settings page view
        include SENTINEL_PLUGIN_DIR . 'admin/views/settings-page.php';
    }
    
    /**
     * Events registry page callback
     * This function is called when someone visits the events page
     */
    public function events_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        // Get event statistics
        $event_stats = Sentinel_Events::get_event_statistics();
        $all_events = Sentinel_Events::get_all_events();
        
        // Include the events page view (we'll create this later)
        echo '<div class="wrap">';
        echo '<h1>Sentinel Event Registry</h1>';
        echo '<p>This page will show all registered events and their configuration.</p>';
        echo '<p>Total events registered: ' . esc_html($event_stats['total_events']) . '</p>';
        echo '</div>';
    }
} 